import React from 'react';
import { FileText, Database, GitMerge, Copy, AlertTriangle, Trash2 } from 'lucide-react';

const SqlDocumentation: React.FC = () => {
  const sqlCode = `-- Enable UUID Extension if not already enabled
CREATE EXTENSION IF NOT EXISTS "pgcrypto";

-- 1. Tabel Laboratorium
CREATE TABLE IF NOT EXISTS laboratorium (
  id UUID DEFAULT gen_random_uuid() PRIMARY KEY,
  nama_lab TEXT NOT NULL,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- 2. Tabel Reagen
CREATE TABLE IF NOT EXISTS reagen (
  id UUID DEFAULT gen_random_uuid() PRIMARY KEY,
  lab_id UUID REFERENCES laboratorium(id) ON DELETE CASCADE,
  nama_reagen TEXT NOT NULL,
  brand TEXT,
  katalog_no TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- 3. Tabel Stok Reagen (Lot)
CREATE TABLE IF NOT EXISTS stok_reagen_lot (
  id UUID DEFAULT gen_random_uuid() PRIMARY KEY,
  reagen_id UUID REFERENCES reagen(id) ON DELETE CASCADE,
  no_lot TEXT NOT NULL,
  tgl_expired DATE NOT NULL,
  stok_sekarang INTEGER NOT NULL DEFAULT 0 CHECK (stok_sekarang >= 0),
  satuan TEXT NOT NULL,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- 4. Tabel Log Pemakaian
CREATE TABLE IF NOT EXISTS log_pemakaian_reagen (
  id UUID DEFAULT gen_random_uuid() PRIMARY KEY,
  lot_id UUID REFERENCES stok_reagen_lot(id) ON DELETE CASCADE,
  jumlah_digunakan INTEGER NOT NULL,
  tgl_input TIMESTAMPTZ DEFAULT NOW(),
  keterangan TEXT
);

-- 5. Tabel Alat
CREATE TABLE IF NOT EXISTS alat (
  id UUID DEFAULT gen_random_uuid() PRIMARY KEY,
  lab_id UUID REFERENCES laboratorium(id) ON DELETE SET NULL,
  nama_alat TEXT NOT NULL,
  merk TEXT,
  tipe TEXT,
  status TEXT DEFAULT 'Tersedia' CHECK (status IN ('Tersedia', 'Dipinjam', 'Rusak', 'Maintenance')),
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- 6. Tabel Peminjaman Alat
CREATE TABLE IF NOT EXISTS peminjaman_alat (
  id UUID DEFAULT gen_random_uuid() PRIMARY KEY,
  alat_id UUID REFERENCES alat(id) ON DELETE CASCADE,
  peminjam TEXT NOT NULL,
  tgl_pinjam TIMESTAMPTZ DEFAULT NOW(),
  tgl_kembali TIMESTAMPTZ,
  status_kembali TEXT DEFAULT 'Sedang Dipinjam'
);

-- 7. Tabel Kalibrasi Alat
CREATE TABLE IF NOT EXISTS kalibrasi_alat (
  id UUID DEFAULT gen_random_uuid() PRIMARY KEY,
  alat_id UUID REFERENCES alat(id) ON DELETE CASCADE,
  tgl_kalibrasi DATE NOT NULL,
  tgl_expired_kalibrasi DATE NOT NULL,
  file_sertifikat_url TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- RLS Policies (Optional but recommended: Allow Anon for demo purposes)
ALTER TABLE laboratorium ENABLE ROW LEVEL SECURITY;
CREATE POLICY "Public Access Lab" ON laboratorium FOR ALL USING (true);

ALTER TABLE reagen ENABLE ROW LEVEL SECURITY;
CREATE POLICY "Public Access Reagen" ON reagen FOR ALL USING (true);

ALTER TABLE stok_reagen_lot ENABLE ROW LEVEL SECURITY;
CREATE POLICY "Public Access Stok" ON stok_reagen_lot FOR ALL USING (true);

ALTER TABLE log_pemakaian_reagen ENABLE ROW LEVEL SECURITY;
CREATE POLICY "Public Access Log" ON log_pemakaian_reagen FOR ALL USING (true);

ALTER TABLE alat ENABLE ROW LEVEL SECURITY;
CREATE POLICY "Public Access Alat" ON alat FOR ALL USING (true);

ALTER TABLE peminjaman_alat ENABLE ROW LEVEL SECURITY;
CREATE POLICY "Public Access Pinjam" ON peminjaman_alat FOR ALL USING (true);

ALTER TABLE kalibrasi_alat ENABLE ROW LEVEL SECURITY;
CREATE POLICY "Public Access Kalibrasi" ON kalibrasi_alat FOR ALL USING (true);
`;

  const dropCode = `-- WARNING: INI AKAN MENGHAPUS SEMUA DATA!
-- PENTING: Jalankan ini jika Anda mengalami error "Foreign Key Constraint" saat menghapus data.
-- Script ini akan menghapus tabel lama secara paksa (CASCADE) dan memungkinkan Anda membuat tabel baru dengan aturan "ON DELETE CASCADE" yang benar.

DROP TABLE IF EXISTS kalibrasi_alat CASCADE;
DROP TABLE IF EXISTS peminjaman_alat CASCADE;
DROP TABLE IF EXISTS log_pemakaian_reagen CASCADE;
DROP TABLE IF EXISTS stok_reagen_lot CASCADE;
DROP TABLE IF EXISTS reagen CASCADE;
DROP TABLE IF EXISTS alat CASCADE;
DROP TABLE IF EXISTS laboratorium CASCADE;
`;

  return (
    <div className="space-y-8 animate-fade-in max-w-4xl">
      <div className="bg-yellow-50 border border-yellow-200 p-4 rounded-lg flex items-start">
        <AlertTriangle className="text-yellow-600 mr-3 flex-shrink-0" />
        <div>
          <h3 className="font-bold text-yellow-800">Troubleshooting: Tidak Bisa Hapus Data?</h3>
          <p className="text-sm text-yellow-700 mt-1">
            Jika Anda mengalami error <strong>"Violates Foreign Key Constraint"</strong> meskipun sudah menggunakan Secret Key, itu karena struktur tabel lama Anda tidak mengizinkan penghapusan berantai.
            <br/><br/>
            <strong>SOLUSI WAJIB:</strong>
            <ol className="list-decimal ml-5 mt-1 space-y-1">
                <li>Copy script <strong>"Reset Database"</strong> (Kotak Merah di bawah).</li>
                <li>Jalankan di Supabase SQL Editor (ini akan menghapus semua data).</li>
                <li>Copy script <strong>"Create Tables"</strong> (Kotak Hitam) dan jalankan ulang.</li>
            </ol>
          </p>
        </div>
      </div>

      <div>
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-2xl font-bold text-slate-800 flex items-center">
            <Database className="mr-3 text-medical-600" /> Create Tables (Setup Awal)
          </h2>
          <button 
            onClick={() => {navigator.clipboard.writeText(sqlCode); alert('SQL Copied to Clipboard!');}}
            className="flex items-center text-sm bg-slate-200 hover:bg-slate-300 px-3 py-1 rounded transition-colors font-medium"
          >
            <Copy size={16} className="mr-2" /> Salin SQL
          </button>
        </div>
        <div className="bg-slate-900 text-slate-50 p-6 rounded-xl font-mono text-xs md:text-sm overflow-x-auto shadow-lg relative h-64">
          <pre>{sqlCode}</pre>
        </div>
      </div>

      <div>
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-2xl font-bold text-rose-700 flex items-center">
            <Trash2 className="mr-3" /> Reset Database (Hapus Semua)
          </h2>
          <button 
            onClick={() => {navigator.clipboard.writeText(dropCode); alert('Drop Code Copied!');}}
            className="flex items-center text-sm bg-rose-100 text-rose-700 hover:bg-rose-200 px-3 py-1 rounded transition-colors font-medium border border-rose-200"
          >
            <Copy size={16} className="mr-2" /> Salin Reset Code
          </button>
        </div>
        <div className="bg-slate-900 text-slate-50 p-6 rounded-xl font-mono text-xs md:text-sm overflow-x-auto shadow-lg relative h-40 border border-rose-200">
          <pre className="text-rose-300">{dropCode}</pre>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
        <div>
          <h2 className="text-xl font-bold text-slate-800 flex items-center mb-4">
            <GitMerge className="mr-3 text-medical-600" /> Logika "Revert Stock"
          </h2>
          <div className="bg-white p-6 rounded-xl border border-slate-200">
            <p className="mb-4 text-slate-600 text-sm">
              Sistem menggunakan logika transaksi klien untuk menjaga integritas data:
            </p>
            
            <div className="relative pl-8 border-l-2 border-slate-200 space-y-6">
              <div className="relative">
                <span className="absolute -left-[41px] bg-blue-500 text-white w-6 h-6 rounded-full flex items-center justify-center text-xs">1</span>
                <h4 className="font-bold text-slate-800 text-sm">Hapus Log</h4>
                <p className="text-xs text-slate-500">Admin menghapus log pemakaian.</p>
              </div>
              
               <div className="relative">
                <span className="absolute -left-[41px] bg-blue-500 text-white w-6 h-6 rounded-full flex items-center justify-center text-xs">2</span>
                <h4 className="font-bold text-slate-800 text-sm">Kembalikan Stok</h4>
                <p className="text-xs text-slate-500">
                  Sistem membaca jumlah pemakaian pada log, lalu menambahkannya kembali ke <code>stok_reagen_lot</code>.
                </p>
              </div>
            </div>
          </div>
        </div>
        
        <div>
           <h2 className="text-xl font-bold text-slate-800 flex items-center mb-4">
            <FileText className="mr-3 text-medical-600" /> Setup Storage
          </h2>
          <div className="bg-white p-6 rounded-xl border border-slate-200 text-sm space-y-2">
              <p>Langkah setup di Supabase Dashboard:</p>
              <ol className="list-decimal list-inside space-y-2 text-slate-600 ml-1">
                  <li>Menu <strong>Storage</strong> -> <strong>New Bucket</strong>.</li>
                  <li>Nama bucket: <code>sertifikat-kalibrasi</code>.</li>
                  <li>Centang <strong>Public Bucket</strong>.</li>
                  <li>Copy URL Endpoint bucket jika diperlukan konfigurasi lanjutan.</li>
              </ol>
          </div>
        </div>
      </div>
    </div>
  );
};

export default SqlDocumentation;