import React, { useState, useEffect } from 'react';
import { db } from '../services/mockSupabase';
import { Lab, Reagen, StokReagenLot, LogPemakaianReagen } from '../types';
import { Plus, Trash2, History, AlertCircle, ChevronRight, Package, Beaker } from 'lucide-react';

const ReagentManagement: React.FC = () => {
  const [labs, setLabs] = useState<Lab[]>([]);
  const [selectedLab, setSelectedLab] = useState<string>('');
  const [reagents, setReagents] = useState<Reagen[]>([]);
  const [selectedReagent, setSelectedReagent] = useState<Reagen | null>(null);
  const [lots, setLots] = useState<StokReagenLot[]>([]);
  const [selectedLot, setSelectedLot] = useState<StokReagenLot | null>(null);
  const [logs, setLogs] = useState<LogPemakaianReagen[]>([]);
  
  // Forms
  const [showAddReagent, setShowAddReagent] = useState(false);
  const [newReagent, setNewReagent] = useState({ nama: '', brand: '', kat_no: '' });
  
  const [showAddLot, setShowAddLot] = useState(false);
  const [newLot, setNewLot] = useState({ no_lot: '', expired: '', stok: 0, satuan: 'ml' });

  const [usageAmount, setUsageAmount] = useState<number>(0);
  const [usageDesc, setUsageDesc] = useState('');

  useEffect(() => {
    db.getLabs().then(setLabs);
  }, []);

  useEffect(() => {
    if (selectedLab) {
      loadReagents();
      setSelectedReagent(null);
      setLots([]);
      setSelectedLot(null);
    }
  }, [selectedLab]);

  const loadReagents = async () => {
    const data = await db.getReagents(selectedLab);
    setReagents(data);
  };

  useEffect(() => {
    if (selectedReagent) {
      loadLots();
      setSelectedLot(null);
    }
  }, [selectedReagent]);

  const loadLots = async () => {
    if (!selectedReagent) return;
    const data = await db.getLots(selectedReagent.id);
    setLots(data);
  };

  useEffect(() => {
    if (selectedLot) {
      loadLogs(selectedLot.id);
    }
  }, [selectedLot]);

  const loadLogs = async (lotId: string) => {
    const data = await db.getLogsByLot(lotId);
    setLogs(data);
  };

  // --- Handlers ---

  const handleCreateReagent = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!selectedLab) return;
    try {
      await db.addReagent({
        lab_id: selectedLab,
        nama_reagen: newReagent.nama,
        brand: newReagent.brand,
        katalog_no: newReagent.kat_no
      });
      setNewReagent({ nama: '', brand: '', kat_no: '' });
      setShowAddReagent(false);
      loadReagents();
    } catch (e: any) {
      alert("Gagal menambah reagen: " + e.message);
    }
  };

  const handleDeleteReagent = async (id: string, e: React.MouseEvent) => {
    e.stopPropagation();
    if (window.confirm("Hapus reagen ini beserta semua stok-nya?")) {
      try {
        await db.deleteReagent(id);
        if (selectedReagent?.id === id) setSelectedReagent(null);
        loadReagents();
      } catch (e: any) {
        alert("Gagal menghapus reagen: " + e.message);
      }
    }
  };

  const handleCreateLot = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!selectedReagent) return;
    try {
      await db.addLot({
        reagen_id: selectedReagent.id,
        no_lot: newLot.no_lot,
        tgl_expired: newLot.expired,
        stok_sekarang: Number(newLot.stok),
        satuan: newLot.satuan
      });
      setNewLot({ no_lot: '', expired: '', stok: 0, satuan: 'ml' });
      setShowAddLot(false);
      loadLots();
    } catch (e: any) {
      alert("Gagal menambah Lot: " + e.message);
    }
  };

  const handleDeleteLot = async (id: string) => {
    if (window.confirm("Hapus Lot ini?")) {
      try {
        await db.deleteLot(id);
        if (selectedLot?.id === id) setSelectedLot(null);
        loadLots();
      } catch (e: any) {
        alert("Gagal menghapus Lot: " + e.message);
      }
    }
  };

  const handleAddUsage = async () => {
    if (!selectedLot || usageAmount <= 0) return;
    try {
      await db.addUsageLog({
        lot_id: selectedLot.id,
        jumlah_digunakan: usageAmount,
        keterangan: usageDesc || 'Pemakaian rutin'
      });
      loadLots();
      const updatedLot = (await db.getLots(selectedReagent!.id)).find(l => l.id === selectedLot.id);
      if(updatedLot) setSelectedLot(updatedLot);
      
      setUsageAmount(0);
      setUsageDesc('');
    } catch (e: any) {
      alert('Error: ' + e.message);
    }
  };

  const handleDeleteLog = async (logId: string) => {
    if (!window.confirm("Yakin hapus log? Stok akan dikembalikan (Revert).")) return;
    try {
      await db.deleteUsageLog(logId);
      loadLots();
      const updatedLot = (await db.getLots(selectedReagent!.id)).find(l => l.id === selectedLot!.id);
      if(updatedLot) setSelectedLot(updatedLot);
    } catch (e: any) {
      alert('Gagal menghapus log: ' + e.message);
    }
  };

  return (
    <div className="space-y-6">
      <h2 className="text-2xl font-bold text-slate-800 flex items-center">
        <span className="bg-medical-100 text-medical-600 p-2 rounded-lg mr-3"><Beaker size={24} /></span>
        Manajemen Reagen & Stok
      </h2>

      {/* Lab Selector & Reagent List */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="md:col-span-1 bg-white p-4 rounded-xl border border-slate-200">
          <label className="block text-sm font-medium text-slate-700 mb-2">Pilih Laboratorium</label>
          <select 
            className="w-full p-2 border border-slate-300 rounded-md focus:ring-2 focus:ring-medical-500 outline-none mb-4"
            value={selectedLab}
            onChange={(e) => setSelectedLab(e.target.value)}
          >
            <option value="">-- Pilih Lab --</option>
            {labs.map(lab => (
              <option key={lab.id} value={lab.id}>{lab.nama_lab}</option>
            ))}
          </select>
          
          <div className="flex justify-between items-center mb-2">
            <h3 className="text-sm font-bold text-slate-600">Daftar Reagen</h3>
            {selectedLab && (
              <button 
                onClick={() => setShowAddReagent(!showAddReagent)}
                className="p-1 bg-medical-50 text-medical-600 rounded hover:bg-medical-100"
                title="Tambah Reagen"
              >
                <Plus size={16} />
              </button>
            )}
          </div>

          {/* Add Reagent Form */}
          {showAddReagent && (
            <form onSubmit={handleCreateReagent} className="bg-slate-50 p-3 rounded-lg mb-4 border border-slate-200 space-y-2">
              <input 
                placeholder="Nama Reagen" 
                required
                className="w-full text-xs p-1 border rounded"
                value={newReagent.nama} onChange={e => setNewReagent({...newReagent, nama: e.target.value})}
              />
              <input 
                placeholder="Brand" 
                className="w-full text-xs p-1 border rounded"
                value={newReagent.brand} onChange={e => setNewReagent({...newReagent, brand: e.target.value})}
              />
              <input 
                placeholder="No. Katalog" 
                className="w-full text-xs p-1 border rounded"
                value={newReagent.kat_no} onChange={e => setNewReagent({...newReagent, kat_no: e.target.value})}
              />
              <div className="flex justify-end gap-2">
                 <button type="button" onClick={() => setShowAddReagent(false)} className="text-xs text-slate-500">Batal</button>
                 <button type="submit" className="text-xs bg-medical-600 text-white px-2 py-1 rounded">Simpan</button>
              </div>
            </form>
          )}
          
          <div className="space-y-2 max-h-[500px] overflow-y-auto">
             {selectedLab && reagents.map(r => (
               <div 
                key={r.id} 
                onClick={() => setSelectedReagent(r)}
                className={`p-3 rounded-md cursor-pointer transition-colors relative group border ${selectedReagent?.id === r.id ? 'bg-medical-50 border-medical-200 text-medical-700' : 'hover:bg-slate-50 border-transparent'}`}
               >
                 <div className="font-medium pr-6">{r.nama_reagen}</div>
                 <div className="text-xs text-slate-500">{r.brand} - {r.katalog_no}</div>
                 <button 
                  onClick={(e) => handleDeleteReagent(r.id, e)}
                  className="absolute top-2 right-2 text-slate-300 hover:text-red-500 opacity-0 group-hover:opacity-100 transition-opacity"
                 >
                   <Trash2 size={14} />
                 </button>
               </div>
             ))}
             {selectedLab && reagents.length === 0 && !showAddReagent && <p className="text-sm text-slate-400 italic">Belum ada data reagen.</p>}
          </div>
        </div>

        {/* Lots & Actions */}
        <div className="md:col-span-3 space-y-6">
          {selectedReagent ? (
            <>
              {/* LOT TABLE */}
              <div className="bg-white p-6 rounded-xl border border-slate-200">
                <div className="flex justify-between items-center mb-4">
                  <h3 className="font-bold text-lg text-slate-700">LOT Reagen: {selectedReagent.nama_reagen}</h3>
                  <button 
                    onClick={() => setShowAddLot(!showAddLot)}
                    className="flex items-center text-sm bg-medical-600 text-white px-3 py-1.5 rounded-lg hover:bg-medical-700"
                  >
                    <Plus size={16} className="mr-1" /> Tambah Lot
                  </button>
                </div>

                {/* Add Lot Form */}
                {showAddLot && (
                  <form onSubmit={handleCreateLot} className="bg-slate-50 p-4 rounded-lg mb-4 border border-slate-200 grid grid-cols-1 md:grid-cols-5 gap-3 items-end">
                    <div>
                      <label className="text-xs font-bold text-slate-500">No. Lot</label>
                      <input required className="w-full text-sm p-2 border rounded" value={newLot.no_lot} onChange={e => setNewLot({...newLot, no_lot: e.target.value})} />
                    </div>
                    <div>
                      <label className="text-xs font-bold text-slate-500">Expired Date</label>
                      <input type="date" required className="w-full text-sm p-2 border rounded" value={newLot.expired} onChange={e => setNewLot({...newLot, expired: e.target.value})} />
                    </div>
                    <div>
                      <label className="text-xs font-bold text-slate-500">Stok Awal</label>
                      <input type="number" required className="w-full text-sm p-2 border rounded" value={newLot.stok} onChange={e => setNewLot({...newLot, stok: Number(e.target.value)})} />
                    </div>
                    <div>
                      <label className="text-xs font-bold text-slate-500">Satuan</label>
                      <select className="w-full text-sm p-2 border rounded" value={newLot.satuan} onChange={e => setNewLot({...newLot, satuan: e.target.value})}>
                        <option value="ml">ml</option>
                        <option value="pcs">pcs</option>
                        <option value="kit">kit</option>
                        <option value="test">test</option>
                        <option value="box">box</option>
                      </select>
                    </div>
                    <div className="flex gap-2">
                       <button type="button" onClick={() => setShowAddLot(false)} className="bg-slate-200 px-3 py-2 rounded text-slate-600 text-sm">Batal</button>
                       <button type="submit" className="bg-medical-600 px-3 py-2 rounded text-white text-sm flex-1">Simpan</button>
                    </div>
                  </form>
                )}

                <div className="overflow-x-auto">
                  <table className="w-full text-sm text-left">
                    <thead className="bg-slate-50 text-slate-700">
                      <tr>
                        <th className="p-3 rounded-tl-lg">No. Lot</th>
                        <th className="p-3">Expired</th>
                        <th className="p-3">Stok Saat Ini</th>
                        <th className="p-3 rounded-tr-lg text-right">Aksi</th>
                      </tr>
                    </thead>
                    <tbody>
                      {lots.length === 0 && <tr><td colSpan={4} className="p-4 text-center text-slate-400">Belum ada data Lot.</td></tr>}
                      {lots.map(lot => (
                        <tr key={lot.id} className={`border-b border-slate-100 ${selectedLot?.id === lot.id ? 'bg-blue-50' : ''}`}>
                          <td className="p-3 font-mono text-slate-600">{lot.no_lot}</td>
                          <td className="p-3">
                            <span className={new Date(lot.tgl_expired) < new Date() ? 'text-red-500 font-bold' : ''}>
                              {new Date(lot.tgl_expired).toLocaleDateString('id-ID')}
                              {new Date(lot.tgl_expired) < new Date() && ' (Expired)'}
                            </span>
                          </td>
                          <td className="p-3 font-bold">{lot.stok_sekarang} {lot.satuan}</td>
                          <td className="p-3 flex justify-end gap-2">
                            <button 
                              onClick={() => setSelectedLot(lot)}
                              className="text-medical-600 hover:text-medical-800 font-medium text-xs px-3 py-1 bg-medical-50 rounded-full border border-medical-100"
                            >
                              Kelola
                            </button>
                            <button 
                              onClick={() => handleDeleteLot(lot.id)}
                              className="text-red-500 hover:text-red-700 p-1"
                              title="Hapus Lot"
                            >
                              <Trash2 size={16} />
                            </button>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </div>

              {/* LOGS & USAGE */}
              {selectedLot && (
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  {/* Input Usage */}
                  <div className="bg-white p-6 rounded-xl border border-slate-200 h-fit">
                    <h3 className="font-bold text-slate-700 mb-4 flex items-center">
                      <ChevronRight className="mr-2" size={16} /> Input Pemakaian
                    </h3>
                    <div className="space-y-4">
                      <div className="bg-yellow-50 p-3 rounded-md text-yellow-800 text-xs flex items-start">
                        <AlertCircle size={16} className="mr-2 mt-0.5 flex-shrink-0" />
                        Jika salah input, hapus log di sebelah kanan untuk mengembalikan stok otomatis.
                      </div>
                      <div>
                        <label className="block text-xs font-medium text-slate-500 uppercase">Jumlah ({selectedLot.satuan})</label>
                        <input 
                          type="number" 
                          className="w-full mt-1 p-2 border rounded-md"
                          value={usageAmount}
                          onChange={(e) => setUsageAmount(Number(e.target.value))}
                        />
                      </div>
                      <div>
                        <label className="block text-xs font-medium text-slate-500 uppercase">Keterangan</label>
                        <input 
                          type="text" 
                          className="w-full mt-1 p-2 border rounded-md"
                          value={usageDesc}
                          onChange={(e) => setUsageDesc(e.target.value)}
                          placeholder="Contoh: Praktikum Hematologi"
                        />
                      </div>
                      <button 
                        onClick={handleAddUsage}
                        disabled={usageAmount <= 0}
                        className="w-full bg-rose-600 hover:bg-rose-700 text-white py-2 rounded-lg font-medium transition-colors disabled:opacity-50"
                      >
                        Kurangi Stok
                      </button>
                    </div>
                  </div>

                  {/* History Log */}
                  <div className="bg-white p-6 rounded-xl border border-slate-200">
                    <h3 className="font-bold text-slate-700 mb-4 flex items-center">
                      <History className="mr-2" size={16} /> Riwayat Log (Audit Trail)
                    </h3>
                    <div className="space-y-3 max-h-64 overflow-y-auto pr-2">
                      {logs.length === 0 && <p className="text-sm text-slate-400">Belum ada riwayat pemakaian.</p>}
                      {logs.map(log => (
                        <div key={log.id} className="flex justify-between items-start p-3 bg-slate-50 rounded-lg border border-slate-100 group">
                          <div>
                            <div className="font-bold text-slate-700">-{log.jumlah_digunakan} {selectedLot.satuan}</div>
                            <div className="text-xs text-slate-500">{new Date(log.tgl_input).toLocaleString('id-ID')}</div>
                            <div className="text-xs text-slate-600 mt-1 italic">"{log.keterangan}"</div>
                          </div>
                          <button 
                            onClick={() => handleDeleteLog(log.id)}
                            className="text-slate-400 hover:text-red-500 opacity-0 group-hover:opacity-100 transition-opacity"
                            title="Hapus Log & Kembalikan Stok"
                          >
                            <Trash2 size={16} />
                          </button>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
              )}
            </>
          ) : (
            <div className="flex flex-col items-center justify-center h-64 bg-white rounded-xl border border-slate-200 border-dashed">
              <Package size={48} className="text-slate-300 mb-2" />
              <p className="text-slate-500">Pilih Laboratorium dan Reagen untuk mengelola stok.</p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default ReagentManagement;