import React, { useState, useEffect } from 'react';
import { db } from '../services/mockSupabase';
import { Lab } from '../types';
import { Database, Plus, Trash2, AlertCircle } from 'lucide-react';

const LabManagement: React.FC = () => {
  const [labs, setLabs] = useState<Lab[]>([]);
  const [newLabName, setNewLabName] = useState('');
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    loadLabs();
  }, []);

  const loadLabs = async () => {
    try {
      const data = await db.getLabs();
      setLabs(data);
    } catch (e: any) {
      setError(e.message);
    }
  };

  const handleAdd = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!newLabName.trim()) return;
    try {
      await db.addLab(newLabName);
      setNewLabName('');
      loadLabs();
      setError(null);
    } catch (e: any) {
      setError('Gagal menambah Lab: ' + e.message);
    }
  };

  const handleDelete = async (id: string) => {
    if (window.confirm("Yakin hapus Lab ini? Semua data reagen terkait akan terhapus otomatis (Cascade).")) {
      try {
        await db.deleteLab(id);
        loadLabs();
        setError(null);
      } catch (e: any) {
        // Handle FK violation gracefully if schema is old
        if (e.message.includes('violates foreign key constraint')) {
          alert("Gagal menghapus: Masih ada data (Reagen/Alat) yang terhubung ke Lab ini. Hapus data terkait terlebih dahulu atau update struktur database.");
        } else {
          alert("Gagal menghapus: " + e.message);
        }
      }
    }
  };

  return (
    <div className="space-y-6">
      <h2 className="text-2xl font-bold text-slate-800 flex items-center">
        <span className="bg-medical-100 text-medical-600 p-2 rounded-lg mr-3"><Database size={24} /></span>
        Data Laboratorium (Master Data)
      </h2>

      {error && (
        <div className="bg-rose-50 border border-rose-200 text-rose-700 p-4 rounded-lg flex items-start gap-3">
          <AlertCircle className="shrink-0 mt-0.5" size={18} />
          <span>{error}</span>
        </div>
      )}

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div className="bg-white p-6 rounded-xl border border-slate-200 h-fit">
          <h3 className="font-bold text-slate-700 mb-4">Tambah Unit Laboratorium</h3>
          <form onSubmit={handleAdd} className="flex gap-2">
            <input 
              type="text"
              value={newLabName}
              onChange={(e) => setNewLabName(e.target.value)}
              placeholder="Nama Laboratorium (Contoh: Lab Parasitologi)"
              className="flex-1 p-2 border border-slate-300 rounded-lg focus:ring-2 focus:ring-medical-500 outline-none"
            />
            <button 
              type="submit" 
              className="bg-medical-600 text-white px-4 py-2 rounded-lg hover:bg-medical-700 flex items-center"
            >
              <Plus size={18} className="mr-1" /> Tambah
            </button>
          </form>
        </div>

        <div className="bg-white p-6 rounded-xl border border-slate-200">
          <h3 className="font-bold text-slate-700 mb-4">Daftar Laboratorium</h3>
          <div className="space-y-2">
            {labs.length === 0 && <p className="text-slate-400 italic">Belum ada data.</p>}
            {labs.map(lab => (
              <div key={lab.id} className="flex justify-between items-center p-3 bg-slate-50 rounded-lg border border-slate-100">
                <span className="font-medium text-slate-700">{lab.nama_lab}</span>
                <button 
                  onClick={() => handleDelete(lab.id)}
                  className="text-slate-400 hover:text-red-600 transition-colors p-1"
                  title="Hapus Lab"
                >
                  <Trash2 size={18} />
                </button>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
};

export default LabManagement;